/**
 * Copyright 2007 Steven Van Impe, Department Of Applied Mathematics And Computer Science, Ghent University
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package taylor;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.font.TextAttribute;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.text.AttributedString;
import java.util.ResourceBundle;
import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Panel that shows a graphical view of the model.
 * Draws the function in the model and its Taylor series expansion.
 *
 * @author Steven Van Impe (steven.vanimpe@ugent.be)
 */
public class TSEView extends JPanel implements ChangeListener {
    
    // model to listen to
    private TSEModel model;
    
    // the transformation from true coordinates to screen coordinates is defined by the following 4 parameters
    
    // translation
    private int tx, ty;
    // scaling - the resolutions (pixels per unit) used for each axis
    private double xres, yres;
    
    // current mouse position
    private int mouseX, mouseY;
    
    /**
     * Create a new panel.
     * Don't forget to set a model afterwards.
     */
    public TSEView() {
        setBackground(Color.WHITE);
        setBorder(BorderFactory.createEtchedBorder());
        
        // catch mouse events
        DrawingMouseListener listener = new DrawingMouseListener();
        addMouseListener(listener);
        addMouseMotionListener(listener);
        addMouseWheelListener(listener);
        
        // rescale the drawing if the user resizes the window
        addComponentListener(new ComponentAdapter() {
            // previous sizes
            private int width = getWidth();
            private int height = getHeight();
            
            // adjust the transformation (scale in the same way the window was scaled) and save sizes for next time
            public void componentResized(ComponentEvent e) {
                if (width > 0 && height > 0 && getWidth() > 0 && getHeight() > 0) {
                    xres *= 1.0 * getWidth() / width;
                    yres *= 1.0 * getHeight() / height;
                    tx *= 1.0 * getWidth() / width;
                    ty *= 1.0 * getHeight() / height;
                    
                    repaint();
                }
                
                width = getWidth();
                height = getHeight();
            }
        });
    }
    
    /**
     * @return the model in use.
     */
    public TSEModel getModel() {
        return model;
    }
    
    /**
     * Set the model to use.
     */
    public void setModel(TSEModel model) {
        if (model != null) {
            if (this.model != null)
                this.model.removeChangeListener(this);
            this.model = model;
            this.model.addChangeListener(this);
            
            repaint();
        }
    }
    
    /**
     * Center the view.
     * This puts the point used for TSE in the center of the panel and scales the view so the visible points range from (-5,-5) to (5,5).
     */
    public void centerView() {
        xres = getWidth() / 10.0;
        yres = getHeight() / 10.0;
        tx = (int) (getWidth() / 2 - xres * model.getPoint());
        ty = (int) (getHeight() / 2 + yres * model.evaluateFunction(model.getPoint()));
        
        repaint();
    }
    
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        
        Graphics2D g2 = (Graphics2D) g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        
        if (model != null) {
            if (! model.isLoading()) {
                drawAxes(g2);
                
                // plot function
                g2.setColor(Color.BLUE);
                g2.setStroke(new BasicStroke(1.25f));
                plotFunction(g2, false);
                
                // plot TSE
                g2.setColor(Color.RED);
                g2.setStroke(new BasicStroke());
                plotFunction(g2, true);
            } else {
                // write a loading message
                AttributedString ats = new AttributedString(ResourceBundle.getBundle("properties/taylor").getString("loadingMessage"));
                ats.addAttribute(TextAttribute.FONT, new Font("Monospaced", Font.BOLD, 12));
                g2.drawString(ats.getIterator(), getWidth() / 2, getHeight() / 2);
            }
        }
    }
    
    // draw X and Y axes
    private void drawAxes(Graphics2D g2) {
        // draw X
        if (ty > 0 && ty < getHeight()) {
            g2.draw(new Line2D.Float(0, ty, getWidth(), ty));
            
            // tickmarks
            
            // the number of units that fit in 100 pixels (rounded up)
            int step = (int) Math.ceil(100 / xres);
            // the first integer on the visible X-axis
            int first = (int) Math.ceil(rx(0));
            // round this number to a multiple of step
            first -= first % step;
            // put tickmarks at first + i * step
            for (int i = 0; dx(first + i * step) < getWidth(); i++)
                tickX(first + i * step, g2);
        }
        // draw Y
        if (tx > 0 && tx < getWidth()) {
            g2.draw(new Line2D.Float(tx, 0, tx, getHeight()));
            
            // tickmarks
            int step = (int) Math.ceil(100 / yres);
            int first = (int) Math.ceil(ry(getHeight() - 1));
            first -= first % step;
            for (int i = 0; dy(first + i * step) > 0; i++)
                tickY(first + i * step, g2);
        }
    }
    
    // show the current mouse position
    private void drawMousePosition(Graphics2D g2) {
        g2.drawString("(" + String.format("%.2f", rx(mouseX)) + "," + String.format("%.2f", ry(mouseY)) + ")", 4 , getHeight() - 8);
    }
    
    // tick the X axis at point x
    private void tickX(int x, Graphics2D g2) {
        g2.draw(new Line2D.Float(dx(x), ty - 3, dx(x), ty + 3));
        g2.drawString(String.format("%d", x), dx(x) + 2, ty + 12);
    }
    
    // tick the Y axis at point y
    private void tickY(int y, Graphics2D g2) {
        g2.draw(new Line2D.Float(tx - 3, dy(y), tx + 3, dy(y)));
        g2.drawString(String.format("%d", y), tx + 2, dy(y) + 12);
    }
    
    // plot a function (taylor == false) or its taylor series (taylor == true)
    private void plotFunction(Graphics2D g2, boolean taylor) {
        GeneralPath p = new GeneralPath();
        p.moveTo(0, taylor ? dy(model.evaluateSeries(rx(0))) : dy(model.evaluateFunction(rx(0))));
        
        for(int x = 1; x < getWidth(); x++)
            p.lineTo(x, taylor ? dy(model.evaluateSeries(rx(x))) : dy(model.evaluateFunction(rx(x))));
        
        g2.draw(p);
    }
    
    /**
     * @return the true abscissa for the pixel with screen abscissa x.
     */
    private double rx(int x) {
        return (x - tx) / xres;
    }
    
    /**
     * @return the screen abscissa for the point with true abscissa x.
     */
    private float dx(double x) {
        return (float) (xres * x + tx);
    }
    
    /**
     * @return the true ordinate for the pixel with screen ordinate y.
     */
    private double ry(int y) {
        return -(y - ty) / yres;
    }
    
    /**
     * @return the screen ordinate for the point with true ordinate y.
     */
    private float dy(double y) {
        return (float) (-yres * y + ty);
    }
    
    /**
     * Update the view.
     */
    public void stateChanged(ChangeEvent e) {
        // center the view if the function or point changed
        if (! model.isLoading() && model.getLastChange() != Change.TERMS)
            centerView();
        // if not, repaint without centering
        else
            repaint();
    }
    
    // handles mouse events for a panel
    private class DrawingMouseListener extends MouseAdapter implements MouseMotionListener, MouseWheelListener {
        
        // values recorded when the mouse is pressed
        private int startTx, startTy;
        private double startXres, startYres;
        private int startX, startY;
        
        // whether or not we are scaling (~shift is down as the mouse is pressed)
        private boolean stretching;
        
        // record values and change cursor
        public void mousePressed(MouseEvent e) {
            startTx = tx;
            startTy = ty;
            startXres = xres;
            startYres = yres;
            startX = e.getX();
            startY = e.getY();
            stretching = e.isShiftDown();
            setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
        }
        
        // return cursor to normal
        public void mouseReleased(MouseEvent e) {
            setCursor(Cursor.getDefaultCursor());
        }
        
        // move or scale (if shift was down) the drawing
        public void mouseDragged(MouseEvent e) {
            if (e.isShiftDown() && stretching) {
                xres = startXres * (e.getX() - tx) / (startX - tx);
                yres = startYres * (e.getY() - ty) / (startY - ty);
                
                // impose lower and upper bounds
                if (xres < 1)
                    xres = 1;
                else if (xres > 10000)
                    xres = 10000;
                
                if (yres < 1)
                    yres = 1;
                else if (yres > 10000)
                    yres = 10000;
            } else if (! stretching) {
                tx = startTx + (e.getX() - startX);
                ty = startTy + (e.getY() - startY);
            }
            
            // save the current mouse position
            mouseX = e.getX();
            mouseY = e.getY();
            
            repaint();
        }
        
        public void mouseMoved(MouseEvent e) {
            // save the current mouse position
            mouseX = e.getX();
            mouseY = e.getY();
        }
        
        // zoom in or out
        public void mouseWheelMoved(MouseWheelEvent e) {
            double x = rx(e.getX());
            double y = ry(e.getY());
            
            xres *= 1 - e.getWheelRotation() * 0.05;
            yres *= 1 - e.getWheelRotation() * 0.05;
            
            // impose lower and upper bounds
            if (xres < 1)
                xres = 1;
            else if (xres > 10000)
                xres = 10000;
            
            if (yres < 1)
                yres = 1;
            else if (yres > 10000)
                yres = 10000;
            
            // make sure the same point is still under the mouse after zooming
            tx -= (int) (dx(x) - e.getX());
            ty -= (int) (dy(y) - e.getY());
            
            repaint();
        }
    }
}
