/**
 * Copyright 2007 Steven Van Impe, Department Of Applied Mathematics And Computer Science, Ghent University
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package taylor;

import java.util.ResourceBundle;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

import org.lsmp.djep.djep.DJep;
import org.lsmp.djep.sjep.PolynomialCreator;
import org.nfunk.jep.Node;
import org.nfunk.jep.ParseException;

/**
 * Enumeration type representing the possible changes in the model (function, point or number of terms).
 */
enum Change {FUNCTION, POINT, TERMS};

/**
 * Model for a function and its Taylor series expansion.
 *
 * @author Steven Van Impe (steven.vanimpe@ugent.be)
 */
public class TSEModel {
    
    // binomial coefficients
    private final int[][] binom = new int[][] {
        {1},
        {1, 1},
        {1, 2, 1},
        {1, 3, 3, 1},
        {1, 4, 6, 4, 1},
        {1, 5, 10, 10, 5, 1},
        {1, 6, 15, 20, 15, 6, 1},
        {1, 7, 21, 35, 35, 21, 7, 1},
        {1, 8, 28, 56, 70, 56, 28, 8, 1},
        {1, 9, 36, 84, 126, 126, 84, 36, 9, 1},
        {1, 10, 45, 120, 210, 252, 210, 120, 45, 10, 1},
        {1, 11, 55, 165, 330, 462, 462, 330, 165, 55, 11, 1},
        {1, 12, 66, 220, 495, 792, 924, 792, 495, 220, 66, 12, 1},
        {1, 13, 78, 286, 715, 1287, 1716, 1716, 1287, 715, 286, 78, 13, 1},
        {1, 14, 91, 364, 1001, 2002, 3003, 3432, 3003, 2002, 1001, 364, 91, 14, 1},
        {1, 15, 105, 455, 1365, 3003, 5005, 6435, 6435, 5005, 3003, 1365, 455, 105, 15, 1}};
    
    // factorials
    private final long[] fac = new long[] {1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L};
    
    // function in use
    private String function;
    // point used for TSE
    private double point;
    // number of terms used in TSE
    private int terms;
    
    // last change in the model
    private Change lastChange;
    
    // whether the model is between states
    private boolean loading;
    
    // coefficients of the taylor polynomial
    private double[] seriesCoeff;
    
    // parser for the function
    private DJep parser;
    // object used to simplify the result returned by the parser
    private PolynomialCreator pc;
    
    // last error encountered
    private String error;
    
    // listeners
    private EventListenerList ll;
    
    /**
     * Create a new model with default values.
     * The default function is sin(x), the default point is x = 0 and the number of terms is 2.
     */
    public TSEModel() {
        // defaults
        function = "sin(x)";
        point = 0.0;
        terms = 2;
        
        seriesCoeff = new double[terms];
        
        parser = new DJep();
        // allow pi and e
        parser.addStandardConstants();
        // allow exp, sin, ...
        parser.addStandardFunctions();
        // allow 2x instead of 2*x
        parser.setImplicitMul(true);
        // this is required by the addStandardDiffRules method
        parser.setAllowUndeclared(true);
        parser.addStandardDiffRules();
        // we only allow variable x
        parser.setAllowUndeclared(false);
        
        // make sure x is known to the parser
        parser.addVariable("x", point);
        // enter the function into the parser
        parser.parseExpression(function);
        
        pc = new PolynomialCreator(parser);
        ll = new EventListenerList();
        
        fillSeriesCoeff();
        
        lastChange = Change.FUNCTION;
        loading = false;
    }
    
    // recalculate the coefficients for the taylor polynomial
    private void fillSeriesCoeff() {
        try {
            // evaluations of the derivatives of f in the point
            double[] deriv = new double[terms];
            
            // set x to the point used for TSE
            parser.addVariable("x", point);
            // parse the function
            // this is separate from parseExpression so the parser will still remember the function we entered with with parseExpression
            Node node = parser.parse(function);
            
            for (int term = 0; term < terms; term++) {
                // evaluate the node
                deriv[term] = (Double) parser.evaluate(node);
                // replace the node by its derivative (which is simplified first)
                node = pc.simplify(parser.differentiate(node, "x"));
            }
            
            // calculate the coefficients by adding up the contributions made by each term in the taylor series expansion
            for (int n = 0; n < terms; n++) {
                seriesCoeff[n] = 0;
                for (int m = n; m < terms; m++)
                    seriesCoeff[n] += deriv[m] / fac[m] * binom[m][n] * Math.pow(-point, m-n);
            }
            
        } catch (ParseException ex) {
            // this shouldn't happen since we already parsed the saved function successfully
            error = parser.getErrorInfo();
        }
    }
    
    /**
     * @return the function in use.
     */
    public synchronized String getFunction() {
        return function;
    }
    
    /**
     * Set the function to use.
     *
     * @return true if the function was accepted, false if the function cannot be used or the requested number of terms was too high for this function.
     */
    public synchronized boolean setFunction(String function) {
        // indicate the model is loading a new state
        loading = true;
        fireStateChanged(new ChangeEvent(this));
        
        String oldFunction = this.function;
        try {
            // try to parse the function
            // if this succeeds, we simplify the result and save its String representation
            this.function = parser.toString(pc.simplify(parser.parse(function)));
            // input the function into the parser
            parser.parseExpression(this.function);
            // update coefficients
            fillSeriesCoeff();
            
            lastChange = Change.FUNCTION;
            return true;
        } catch (ParseException e) {
            // if parsing fails, we save the error
            // since we used parse instead of parseExpression, our parser was unaffected
            error = ResourceBundle.getBundle("properties/taylor").getString("invalidFunction") + parser.getErrorInfo();
            
            return false;
        } catch (ArrayIndexOutOfBoundsException e) {
            // this catches errors during fillEvaluations with functions the DJep cannot process
            error = ResourceBundle.getBundle("properties/taylor").getString("cannotUseFunction");
            
            // restore previous state
            this.function = oldFunction;
            parser.parseExpression(this.function);
            fillSeriesCoeff();
            
            return false;
        } catch (OutOfMemoryError e) {
            // heap space might run out during fillEvaluations if too many terms are requested
            error = ResourceBundle.getBundle("properties/taylor").getString("outOfMemory");
            
            // if this happens, we keep the requested function but reduce the number of terms to a safe default (2)
            terms = 2;
            seriesCoeff = new double[terms];
            fillSeriesCoeff();
            
            lastChange = Change.FUNCTION;
            return false;
        } finally {
            // inform listeners that loading has finished
            loading = false;
            fireStateChanged(new ChangeEvent(this));
        }
    }
    
    /**
     * @return the point used for Taylor series expansion.
     */
    public synchronized double getPoint() {
        return point;
    }
    
    /**
     * Set the point to use for Taylor series expansion.
     */
    public synchronized void setPoint(double point) {
        // indicate the model is loading a new state
        loading = true;
        fireStateChanged(new ChangeEvent(this));
        
        this.point = point;
        fillSeriesCoeff();
        
        lastChange = Change.POINT;
        loading = false;
        fireStateChanged(new ChangeEvent(this));
    }
    
    /**
     * @return the number of terms in the Taylor series.
     */
    public synchronized int getNumberOfTerms() {
        return terms;
    }
    
    /**
     * Set the number of terms in the Taylor series.
     *
     * @return whether the program can handle this many terms for the current function.
     */
    public synchronized boolean setNumberOfTerms(int terms) {
        // indicate the model is loading a new state
        loading = true;
        fireStateChanged(new ChangeEvent(this));
        
        int oldTerms = this.terms;
        try {
            // make sure terms is >= 1 and <= 16
            this.terms = terms < 1 ? 1 : (terms > 16 ? 16 : terms);
            seriesCoeff = new double[this.terms];
            fillSeriesCoeff();
            
            lastChange = Change.TERMS;
            return true;
        } catch (OutOfMemoryError e) {
            error = ResourceBundle.getBundle("properties/taylor").getString("outOfMemory");
            
            // restore previous state
            this.terms = oldTerms;
            seriesCoeff = new double[this.terms];
            fillSeriesCoeff();
            
            return false;
        } finally {
            // inform listeners that loading has finished
            loading = false;
            fireStateChanged(new ChangeEvent(this));
        }
    }
    
    /**
     * @return the coefficient of x<sup>n</sup> in the taylor series expansion.
     */
    public double getSeriesCoeff(int n) {
        return (n >= 0 && n < terms) ? seriesCoeff[n] : Double.NaN;
    }
    
    /**
     * @return the last change in the model.
     */
    public synchronized Change getLastChange() {
        return lastChange;
    }
    
    /**
     * @return whether the model is between states. If true, trying to get the state of the model will cause delays.
     */
    public boolean isLoading() {
        return loading;
    }
    
    /**
     * @return the last error that occurred.
     */
    public synchronized String getError() {
        return error;
    }
    
    /**
     * @return the function evaluated in the given point x.
     */
    public synchronized double evaluateFunction(double x) {
        parser.addVariable("x", x);
        return parser.getValue();
    }
    
    /**
     * @return the Taylor series evaluated in the given point x.
     */
    public synchronized double evaluateSeries(double x) {
        double result = seriesCoeff[0];
        for (int term = 1; term < terms; term++)
            result += Math.pow(x, term) * seriesCoeff[term];
        return result;
    }
    
    /**
     * Add a ChangeListener.
     */
    public void addChangeListener(ChangeListener l) {
        ll.add(ChangeListener.class, l);
    }
    
    /**
     * Remove a ChangeListener.
     */
    public void removeChangeListener(ChangeListener l) {
        ll.remove(ChangeListener.class, l);
    }
    
    /**
     * Notify all ChangeListeners.
     */
    public void fireStateChanged(ChangeEvent e) {
        ChangeListener[] listeners = ll.getListeners(ChangeListener.class);
        for (ChangeListener l : listeners)
            l.stateChanged(e);
    }
}
