/**
 * Copyright 2007 Steven Van Impe, Department Of Applied Mathematics And Computer Science, Ghent University
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package taylor;

import java.text.DecimalFormat;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.nfunk.jep.JEP;

/**
 * Main window. Contains both view and controller.
 *
 * @author Steven Van Impe (steven.vanimpe@ugent.be)
 */
public class TSEFrame extends javax.swing.JFrame implements ChangeListener {
    
    // model to use
    private TSEModel model;
    
    // parser for the point (this allows the input of expressions with standard operators/functions/constants)
    private JEP jep;
    
    /**
     * Create a new frame.
     */
    public TSEFrame() {
        model = new TSEModel();
        model.addChangeListener(this);
        
        jep = new JEP();
        // allow implicit multiplication and add standard functions/constants
        jep.setImplicitMul(true);
        jep.addStandardConstants();
        jep.addStandardFunctions();
        
        initComponents();
        
        // the model was set during initComponents, but we still have to center the view
        tSEView1.centerView();
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        tSEView1 = new taylor.TSEView();
        jLabel1 = new javax.swing.JLabel();
        jTextField1 = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        jSpinner1 = new javax.swing.JSpinner();
        jLabel3 = new javax.swing.JLabel();
        jTextField2 = new javax.swing.JTextField();
        jLabel4 = new javax.swing.JLabel();
        jTextField3 = new javax.swing.JTextField();

        setTitle(java.util.ResourceBundle.getBundle("properties/taylor").getString("windowTitle"));
        tSEView1.setModel(model);
        org.jdesktop.layout.GroupLayout tSEView1Layout = new org.jdesktop.layout.GroupLayout(tSEView1);
        tSEView1.setLayout(tSEView1Layout);
        tSEView1Layout.setHorizontalGroup(
            tSEView1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(0, 406, Short.MAX_VALUE)
        );
        tSEView1Layout.setVerticalGroup(
            tSEView1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(0, 406, Short.MAX_VALUE)
        );

        jLabel1.setText(java.util.ResourceBundle.getBundle("properties/taylor").getString("pointLabel"));

        jTextField1.setColumns(5);
        jTextField1.setText("" + model.getPoint());
        jTextField1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jTextField1ActionPerformed(evt);
            }
        });

        jLabel2.setText(java.util.ResourceBundle.getBundle("properties/taylor").getString("termsLabel"));

        jSpinner1.setModel(new SpinnerNumberModel(model.getNumberOfTerms(), 1, 16, 1));
        jSpinner1.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSpinner1StateChanged(evt);
            }
        });

        jLabel3.setText(java.util.ResourceBundle.getBundle("properties/taylor").getString("functionLabel"));

        jTextField2.setForeground(new java.awt.Color(0, 0, 255));
        jTextField2.setText(model.getFunction());
        jTextField2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jTextField2ActionPerformed(evt);
            }
        });

        jLabel4.setText(java.util.ResourceBundle.getBundle("properties/taylor").getString("seriesLabel"));

        jTextField3.setEditable(false);
        jTextField3.setForeground(new java.awt.Color(255, 0, 0));
        jTextField3.setText(buildSeries());

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .add(12, 12, 12)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, jLabel4)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, jLabel1)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, jLabel3))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(layout.createSequentialGroup()
                        .add(jTextField1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 180, Short.MAX_VALUE)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jLabel2)
                        .add(12, 12, 12)
                        .add(jSpinner1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                    .add(jTextField3, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 326, Short.MAX_VALUE)
                    .add(jTextField2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 326, Short.MAX_VALUE))
                .addContainerGap())
            .add(tSEView1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                .add(tSEView1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jLabel3)
                    .add(jTextField2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jLabel4)
                    .add(jTextField3, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jLabel2)
                    .add(jSpinner1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jTextField1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jLabel1))
                .addContainerGap())
        );
        pack();
    }// </editor-fold>//GEN-END:initComponents
    
    // set the point
    private void jTextField1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jTextField1ActionPerformed
        new Thread(new Runnable() {
            public void run() {
                // try to parse the input
                jep.parseExpression(jTextField1.getText());
                if (! jep.hasError())
                    model.setPoint(jep.getValue());
                // make sure the textfield shows the correct value (either the parsed value (if the input was valid) or the previous value (if the input was invalid))
                jTextField1.setText("" + model.getPoint());
            }
        }).start();
    }//GEN-LAST:event_jTextField1ActionPerformed
    
    // change the number of terms
    private void jSpinner1StateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSpinner1StateChanged
        new Thread(new Runnable() {
            public void run() {
                // try to set the requested number of terms
                if (! model.setNumberOfTerms((Integer) jSpinner1.getValue())) {
                    // if this failed, show the error and reset the spinner to its previous value
                    showError();
                    jSpinner1.setValue(model.getNumberOfTerms());
                }
            }
        }).start();
    }//GEN-LAST:event_jSpinner1StateChanged
    
    // set the function
    private void jTextField2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jTextField2ActionPerformed
        new Thread(new Runnable() {
            public void run() {
                // try to set the function
                if (! model.setFunction(jTextField2.getText()))
                    showError();
                // update the values (the number of terms can be changed as well, in case the previous value was too high for the new function)
                jTextField2.setText(model.getFunction());
                jSpinner1.setValue(model.getNumberOfTerms());
            }
        }).start();
    }//GEN-LAST:event_jTextField2ActionPerformed
    
    // show a dialog window with the error reported by the model
    private void showError() {
        JOptionPane.showMessageDialog(this, model.getError(), java.util.ResourceBundle.getBundle("properties/taylor").getString("errorWindowTitle"), JOptionPane.ERROR_MESSAGE);
    }
    
    // build a string for the taylor series (based on the coefficients in the model)
    private String buildSeries() {
        StringBuffer sb = new StringBuffer();
        // formats double with scientific notation
        DecimalFormat df = new DecimalFormat("0.###E0");
        
        int i = 0;
        
        // skip zeros
        while (i < model.getNumberOfTerms() && model.getSeriesCoeff(i) == 0)
            i++;
        
        // return "0.0" if there are no non-zero terms
        if (i == model.getNumberOfTerms())
            sb.append("0.0");
        else {
            // add the first non-zero term
            if (i == 0)
                sb.append(df.format(model.getSeriesCoeff(i)));
            else
                sb.append(df.format(model.getSeriesCoeff(i)) + " * x^" + i);
            i++;
            
            // add remaining non-zero terms
            while(i < model.getNumberOfTerms()) {
                if (model.getSeriesCoeff(i) != 0)
                    sb.append((model.getSeriesCoeff(i) > 0 ? " +" : " ") + df.format(model.getSeriesCoeff(i)) + " * x^" + i);
                i++;
            }
        }
        
        return sb.toString();
    }
    
    // update the taylor series
    public void stateChanged(ChangeEvent e) {
        jTextField3.setText(buildSeries());
    }
    
    /**
     * Main program. Creates a frame and shows it.
     */
    public static void main(String args[]) {
        TSEFrame frame = new TSEFrame();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setVisible(true);
    }
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JSpinner jSpinner1;
    private javax.swing.JTextField jTextField1;
    private javax.swing.JTextField jTextField2;
    private javax.swing.JTextField jTextField3;
    private taylor.TSEView tSEView1;
    // End of variables declaration//GEN-END:variables
}
